// openai_stream.hpp
#pragma once
#include <curl/curl.h>
#include <nlohmann/json.hpp>
#include <functional>
#include <string>
#include <stdexcept>
#include <cstdlib>

namespace openai {

using json = nlohmann::json;

/*  对外回调签名
    参数说明：
    response : 本次解析后得到的完整 JSON 块
               必含字段：
                 "id"       -> 本次 chunk 的 id
                 "model"    -> 本次请求的 model
                 "delta"    -> 本次新增的 delta 字符串
                 "finish_reason" -> 可能为空字符串
    当 finish_reason 不为空时表示流已结束，可据此做 flush/close 等动作
*/
using StreamCallback = std::function<void(const json& response)>;

struct OpenAIError : public std::runtime_error {
    explicit OpenAIError(const std::string& msg) : std::runtime_error(msg) {}
};

// ---------- 内部实现 ----------
namespace detail {

struct UserCtx {
    std::string lineBuf;
    StreamCallback outerCb;
};

static size_t writeCB(char* ptr, size_t size, size_t n, void* userdata) {
    size_t len = size * n;
    auto* ctx = static_cast<UserCtx*>(userdata);

    for (size_t i = 0; i < len; ++i) {
        char c = ptr[i];
        ctx->lineBuf.push_back(c);
        if (c == '\n') {
            std::string line = std::move(ctx->lineBuf);
            ctx->lineBuf.clear();
            // 去掉尾部 \r\n
            while (!line.empty() && (line.back() == '\r' || line.back() == '\n'))
                line.pop_back();

            if (line.rfind("data: ", 0) == 0) {
                std::string payload = line.substr(6);
                if (payload == "[DONE]") continue;

                try {
                    auto raw = json::parse(payload);
                    json out;
                    out["id"]   = raw.value("id", "");
                    out["model"]= raw.value("model", "");
                    out["delta"]= "";
                    out["finish_reason"] = "";

                    if (raw.contains("choices") && !raw["choices"].empty()) {
                        const auto& choice = raw["choices"][0];
                        if (choice.contains("delta") && choice["delta"].contains("content"))
                            out["delta"] = choice["delta"]["content"].get<std::string>();
                        if (choice.contains("finish_reason") && !choice["finish_reason"].is_null())
                            out["finish_reason"] = choice["finish_reason"].get<std::string>();
                    }
                    ctx->outerCb(out);
                } catch (...) { /* 忽略格式错误的行 */ }
            }
        }
    }
    return len;
}

} // namespace detail

// ---------- 对外接口 ----------
inline void chat(const std::string& model,
                 const std::string& userText,
                 const StreamCallback& cb)
{
    const char* apiKey  = std::getenv("OPENAI_API_KEY");
    const char* baseUrl = std::getenv("OPENAI_API_BASE");
    if (!apiKey)  throw OpenAIError("OPENAI_API_KEY not set");
    if (!baseUrl) throw OpenAIError("OPENAI_API_BASE not set");

    std::string url = std::string(baseUrl) + "/chat/completions";

    json body = {
        {"model", model},
        {"messages", json::array({{{"role","user"},{"content",userText}}})},
        {"stream", true}
    };
    std::string bodyStr = body.dump();

    CURL* curl = curl_easy_init();
    if (!curl) throw OpenAIError("curl_easy_init failed");

    struct curl_slist* h = nullptr;
    h = curl_slist_append(h, "Content-Type: application/json");
    h = curl_slist_append(h, ("Authorization: Bearer " + std::string(apiKey)).c_str());
    h = curl_slist_append(h, "Accept: text/event-stream");

    detail::UserCtx ctx{{}, cb};
    curl_easy_setopt(curl, CURLOPT_URL, url.c_str());
    curl_easy_setopt(curl, CURLOPT_HTTPHEADER, h);
    curl_easy_setopt(curl, CURLOPT_POST, 1L);
    curl_easy_setopt(curl, CURLOPT_POSTFIELDS, bodyStr.c_str());
    curl_easy_setopt(curl, CURLOPT_WRITEFUNCTION, detail::writeCB);
    curl_easy_setopt(curl, CURLOPT_WRITEDATA, &ctx);
    curl_easy_setopt(curl, CURLOPT_TIMEOUT, 0L);
    curl_easy_setopt(curl, CURLOPT_TCP_KEEPALIVE, 1L);

    CURLcode res = curl_easy_perform(curl);
    curl_slist_free_all(h);
    curl_easy_cleanup(curl);
    if (res != CURLE_OK)
        throw OpenAIError(std::string("curl: ") + curl_easy_strerror(res));
}

} // namespace openai